const axios = require('axios');


//Normaliza CHARG para evitar errores de comparación
 
function normalizarCharg(valor) {
    if (valor === null || valor === undefined) return '';
    return String(valor).trim();
}

 //Anula movimientos 102 (devoluciones) que correspondan a 101 (entradas)

function anularMovimientosDevolucion(movimientos) {

    const TOLERANCIA = 0.01;

    // Normalizar todos los movimientos
    const movs = movimientos.map(m => ({
        ...m,
        _id: Symbol(), // clave única temporal
        BWART: Number(m.BWART),
        ERFMG: Number(m.ERFMG),
        CHARG: normalizarCharg(m.CHARG),
        MATNR: String(m.MATNR)
    }));

    const mov101 = movs.filter(m => m.BWART === 101);
    const mov102 = movs.filter(m => m.BWART === 102);

    const eliminados = new Set();

    for (const dev of mov102) {
        const original = mov101.find(org =>
            !eliminados.has(org._id) &&
            org.CHARG === dev.CHARG &&
            org.MATNR === dev.MATNR &&
            Math.abs(org.ERFMG - dev.ERFMG) <= TOLERANCIA
        );

        if (original) {
            eliminados.add(original._id);
            eliminados.add(dev._id);
        }
    }

    return movs.filter(m => !eliminados.has(m._id));
}

const consultarMB51 = async (planilla, tokenSAP) => {

    const { RFC_BASE } = process.env;

    try {
        const rfcPayload = {
            INPUT: {
                I_XBLNR: planilla,
                ITR_WERKS: [
                    {
                        SIGN: "I",
                        OPTION: "EQ",
                        LOW: "C114",
                        HIGH: ""
                    }
                ],
                // ============================================
                // TIPOS DE MOVIMIENTO: 101-102 (entradas), 541-543 (salidas)
                // ============================================
                ITR_BWART: [
                    {
                        SIGN: "I",
                        OPTION: "BT",
                        LOW: "101",
                        HIGH: "102"
                    },
                    {
                        SIGN: "I",
                        OPTION: "BT",
                        LOW: "541",
                        HIGH: "543"
                    }
                ],
                ITR_MATNR: [
                    {
                        SIGN: "I",
                        OPTION: "EQ",
                        LOW: "CE_GRANEL"
                    }
                ]
            },
            RFC: "ZMF_GET_MB51",
            getbase: false
        }

        const mb51Response = await axios.post(`${RFC_BASE}`,
            rfcPayload, {
            headers: {
                Authorization: `Bearer ${tokenSAP}`,
                'Content-Type': 'application/json',
            }
        });

        const rawResultsData = mb51Response.data?.data?.ZMF_GET_MB51?.OUTPUT?.ET_DATA?.item || [];
        const rawResults = Array.isArray(rawResultsData) ? rawResultsData : (rawResultsData ? [rawResultsData] : []);

        console.log(`MB51 OK para ${planilla} -> elementos brutos: ${rawResults.length}`);

        // === APLICAR ANULACIÓN AUTOMÁTICA ===
        const datosFiltrados = anularMovimientosDevolucion(rawResults);

        console.log(`MB51 LIMPIO para ${planilla} -> movimientos finales: ${datosFiltrados.length}`);

        return datosFiltrados;

    } catch (error) {
        console.error(`Error al consultar MB51 para ${planilla}:`, error.message);
        return [];
    }
};

module.exports = {
    consultarMB51
}