const axios = require('axios');
const {
    Proveedor, RecepcionDetail, RecepcionHeader,
    Especie, Variedad, ProcesoHeader, ProcesoIn,
    ProcesoOut, Predio, Temporada, ProcesoMaterial,
    ProcesoCalibreTipo, Logs
} = require('../models/index')
const { sequelize } = require('../config/database');
const { raw } = require('mysql2');

const { consultarMB51 } = require('../helpers/consultarMB51.helper')
const { enriquecerDatosConDetallesLote } = require('../helpers/enriquecerDatos.helper');
const { prepararDatosParaInsercion } = require('../helpers/prepararDatos.helper');

const sincronizar_rs_reina_sur = async (req, res) => {
    const { user_reina_sur, pass_reina_sur, urlBase, rutaLogin, rutaRecepcionMP } = process.env;

    try {
        // --- LOGIN ---
        const loginResponse = await axios.post(`${urlBase}${rutaLogin}`, {
            username: user_reina_sur,
            password: pass_reina_sur
        });

        const token = loginResponse.data.token;
        if (!token) {
            return res.status(401).json({ error: 'Token no recibido.' });
        }

        // --- OBTENER RECEPCIONES ---
        const recepcionResponse = await axios.get(`${urlBase}${rutaRecepcionMP}`, {
            headers: { Authorization: `Bearer ${token}` }
        });

        if (!recepcionResponse) {
            return res.status(404).json({ error: 'No se obtuvo respuesta de recepción.' });
        }

        const rawResultsData = recepcionResponse.data;
        const rawResults = Array.isArray(rawResultsData) ? rawResultsData : (rawResultsData ? [rawResultsData] : []);

        // --- AGRUPAR POR GUÍA BASE + PROVEEDOR ---
        const groupedByGuide = rawResults.reduce((acc, row) => {
            // Extraer guía base (quitar decimales)
            const guiaBase = row.guia_productor ? row.guia_productor.split('.')[0] : 'SIN_GUIA';

            // Normalizar nombre proveedor
            const proveedorNombre = (row.descripcion_predio || '').trim().toUpperCase();

            // Crear clave única: guía_base + proveedor
            const groupKey = `${guiaBase}_${proveedorNombre}`;

            if (!acc[groupKey]) {
                acc[groupKey] = {
                    // Datos del header (tomamos del primer registro)
                    guia_base: guiaBase,
                    fecha_guia_productor: row.fecha_guia_productor,
                    fecha_recepcion: row.fecha_recepcion,
                    especie: row.especie,
                    descripcion_predio: row.descripcion_predio,
                    guias_originales: [],
                    // Array para acumular detalles
                    detalles: []
                };
            }

            // Agregar guía original si no existe
            if (!acc[groupKey].guias_originales.includes(row.guia_productor)) {
                acc[groupKey].guias_originales.push(row.guia_productor);
            }

            // Agregar detalle
            acc[groupKey].detalles.push({
                lote: row.lote,
                kilos_netos: parseFloat(row.kilos_netos) || 0,
                variedad: (row.variedad || '').trim(),
                especie: (row.especie || '').trim()
            });

            return acc;
        }, {});

        const processSummary = [];

        // --- PROCESAR CADA GUÍA AGRUPADA ---
        for (const [groupKey, data] of Object.entries(groupedByGuide)) {
            const t = await sequelize.transaction();
            try {
                // --- Normalizar datos del header ---
                const registroRecepcion = data.fecha_guia_productor && !isNaN(new Date(data.fecha_guia_productor))
                    ? new Date(data.fecha_guia_productor)
                    : null;

                const fechaRecepcion = data.fecha_recepcion && !isNaN(new Date(data.fecha_recepcion))
                    ? new Date(data.fecha_recepcion)
                    : null;

                // Usar la guía base para el registro
                const guiaRecepcion = data.guia_base;
                const paisId = 1;
                const temporadaId = 1;
                // const predioId = 1894;

                // --- Buscar especie ---
                let especieId = null;
                if (data.especie) {
                    const especieEncontrada = await Especie.findOne({
                        where: sequelize.where(
                            sequelize.fn('upper', sequelize.col('gen_nombre_especie')),
                            data.especie.toUpperCase()
                        ),
                        attributes: ['gen_idtbl_especie'],
                        transaction: t
                    });
                    if (especieEncontrada) {
                        especieId = especieEncontrada.gen_idtbl_especie;
                    }
                }

                if (!especieId) {
                    console.warn(`Especie no encontrada: ${data.especie}, guía omitida`);
                    await t.rollback();
                    processSummary.push({
                        error: `Especie no encontrada (${data.especie})`,
                        guia_base: guiaRecepcion,
                        guias_originales: data.guias_originales,
                        proveedor: data.descripcion_predio
                    });
                    continue;
                }

                // --- Buscar proveedor ---
                let proveedorId = null;
                let predioId = null;
                const proveedorNombre = (data.descripcion_predio || '').trim();

                if (proveedorNombre) {
                    const proveedorEncontrado = await Proveedor.findOne({
                        where: sequelize.where(
                            sequelize.fn('upper', sequelize.col('gen_nombre_proveedor')),
                            proveedorNombre.toUpperCase()
                        ),
                        attributes: ['gen_idtbl_proveedor'],
                        transaction: t
                    });

                    if (proveedorEncontrado) {
                        proveedorId = proveedorEncontrado.gen_idtbl_proveedor;

                        // Buscar predio usando el proveedor
                        const predioEncontrado = await Predio.findOne({
                            where: { gen_fk_tbl_proveedor_idtbl_proveedor: proveedorId },
                            attributes: ['gen_idtbl_predio'],
                            transaction: t
                        });

                        if (predioEncontrado) {
                            predioId = predioEncontrado.gen_idtbl_predio;
                        } else {
                            console.warn(`Predio no encontrado para proveedor ${proveedorNombre}, guía omitida`);
                            await t.rollback();
                            processSummary.push({
                                error: `Predio no encontrado para proveedor (${proveedorNombre})`,
                                guia_base: guiaRecepcion,
                                guias_originales: data.guias_originales,
                                proveedor: proveedorNombre
                            });
                            continue;
                        }
                    } else {
                        console.warn(`Proveedor no encontrado: ${proveedorNombre}, guía omitida`);
                        await t.rollback();
                        processSummary.push({
                            error: `Proveedor no encontrado (${proveedorNombre})`,
                            guia_base: guiaRecepcion,
                            guias_originales: data.guias_originales,
                            proveedor: proveedorNombre
                        });
                        continue;
                    }
                }


                // --- Verificar si ya existe ---
                const recepcionExistente = await RecepcionHeader.findOne({
                    where: {
                        gen_guia_recepcion: guiaRecepcion,
                        gen_fk_tbl_proveedor_idtbl_proveedor: proveedorId,
                        gen_tbl_especie_gen_idtbl_especie: especieId
                    },
                    transaction: t
                });

                if (recepcionExistente) {
                    console.warn(`Recepción duplicada: guía ${guiaRecepcion}, proveedor ${proveedorId}, especie ${especieId}`);
                    await t.rollback();
                    processSummary.push({
                        error: `Recepción duplicada (guía: ${guiaRecepcion}, proveedor: ${data.descripcion_predio}, especie: ${data.especie})`,
                        guia_base: guiaRecepcion,
                        guias_originales: data.guias_originales,
                        proveedor: data.descripcion_predio
                    });
                    continue;
                }

                // --- Crear recepción header ---
                const headerInserted = await RecepcionHeader.create({
                    gen_registro_recepcion: registroRecepcion,
                    gen_fecha_recepcion: fechaRecepcion,
                    gen_guia_recepcion: guiaRecepcion,
                    gen_fk_tbl_pais_id_pais: paisId,
                    gen_fk_tbl_temporada_idtbl_temporada: temporadaId,
                    gen_fk_tbl_predio_idtbl_predio: predioId,
                    gen_tbl_especie_gen_idtbl_especie: especieId,
                    gen_fk_tbl_recepcion_calidad_idtbl_recepcion_calidad: null,
                    url_view: null,
                    gen_fk_tbl_proveedor_idtbl_proveedor: proveedorId
                }, { transaction: t });

                if (!headerInserted?.gen_idtbl_recepcion) {
                    throw new Error('No se obtuvo ID del header insertado.');
                }

                // --- Agrupar detalles por variedad ---
                const detallesPorVariedad = {};
                for (const detalle of data.detalles) {
                    const variedadKey = (detalle.variedad || 'SIN_VARIEDAD').toUpperCase();
                    if (!detallesPorVariedad[variedadKey]) detallesPorVariedad[variedadKey] = [];
                    detallesPorVariedad[variedadKey].push(detalle);
                }

                let detallesInsertados = 0;
                for (const [variedadKey, detalles] of Object.entries(detallesPorVariedad)) {
                    // Buscar variedad
                    let variedadId = null;
                    if (variedadKey !== 'SIN_VARIEDAD') {
                        const variedadEncontrada = await Variedad.findOne({
                            where: sequelize.where(
                                sequelize.fn('upper', sequelize.col('gen_nombre_variedad')),
                                variedadKey
                            ),
                            attributes: ['idtbl_variedad'],
                            transaction: t
                        });
                        if (variedadEncontrada) {
                            variedadId = variedadEncontrada.idtbl_variedad;
                        }
                    }

                    if (!variedadId) {
                        // Si no existe la variedad, omitir ese grupo de lotes
                        continue;
                    }

                    // --- Sumar todos los kilos de los detalles ---
                    const totalKilos = detalles.reduce((sum, detalle) => sum + detalle.kilos_netos, 0);

                    // --- Concatenar lotes únicos ---
                    const lotesUnicos = [...new Set(detalles.map(detalle => detalle.lote).filter(Boolean))];
                    const lotesConcatenados = lotesUnicos.join(', ') || null;

                    // --- Crear UN SOLO detalle con la suma de kilos ---
                    const detailInserted = await RecepcionDetail.create({
                        gen_fk_tbl_recepcion_header_idtbl_recepcion: headerInserted.gen_idtbl_recepcion,
                        gen_batch_recepcion_detail: lotesConcatenados,
                        gen_cantidad_recepcion_detail: totalKilos > 0 ? String(totalKilos.toFixed(2)) : null,
                        gen_um_recepcion_detail: 'KG',
                        gen_material_recepcion_detail: 'CE_GRANEL',
                        gen_fk_tbl_predio_idtbl_predio: predioId,
                        fk_tbl_variedad_idtbl_variedad: variedadId
                    }, { transaction: t });

                    detallesInsertados++;
                }

                if (detallesInsertados === 0) {
                    await t.rollback();
                    processSummary.push({
                        error: `No se insertó ningún detalle válido (variedades no encontradas)`,
                        guia_base: guiaRecepcion,
                        guias_originales: data.guias_originales,
                        proveedor: data.descripcion_predio
                    });
                    continue;
                }

                await t.commit();

                processSummary.push({
                    success: true,
                    guia_base: guiaRecepcion,
                    guias_originales_agrupadas: data.guias_originales,
                    headerId: headerInserted.gen_idtbl_recepcion,
                    proveedorId,
                    proveedor: data.descripcion_predio,
                    detallesInsertados
                });

            } catch (errInner) {
                await t.rollback();
                processSummary.push({
                    error: errInner.message || String(errInner),
                    guia_base: data.guia_base,
                    guias_originales: data.guias_originales,
                    proveedor: data.descripcion_predio
                });
            }
        }

        return res.status(200).json({
            message: 'ok',
            processed: processSummary,
            totalGuiasAgrupadas: Object.keys(groupedByGuide).length,
            totalRegistrosOriginales: rawResults.length
        });

    } catch (error) {
        console.error('sincronizar_rs_reina_sur error:', error);
        return res.status(500).json({
            error: error?.message || error
        });
    }
};

const sincronizar_procesos_etl_cerezas = async (req, res) => {
    const { user_reina_sur, pass_reina_sur, urlBase, rutaLogin, rutaProcesos, RFC_BASE, RFC_USER, RFC_PASS, RFC_BASE_URL_LOGIN } = process.env;

    try {
        // --- LOGIN ---
        const loginResponse = await axios.post(`${urlBase}${rutaLogin}`, {
            username: user_reina_sur,
            password: pass_reina_sur
        });

        const token = loginResponse.data.token;
        if (!token) {
            return res.status(401).json({ error: 'Token no recibido.' });
        }

        // Obtener token SAP para RFC
        const loginSAP = await axios.post(RFC_BASE_URL_LOGIN, {
            JCO_USER: RFC_USER,
            JCO_PASSWD: RFC_PASS
        });

        const tokenSAP = loginSAP.data.token;
        if (!tokenSAP) {
            return res.status(401).json({ error: 'Token SAP no recibido.' });
        }

        // --- OBTENER PROCESOS ---
        const procesosResponse = await axios.get(`${urlBase}${rutaProcesos}`, {
            headers: { Authorization: `Bearer ${token}` },
            params: { Estado_Proceso: 'Cerrado' }
        });

        if (!procesosResponse) {
            return res.status(404).json({ error: 'No se obtuvo respuesta de procesos.' });
        }

        const rawResultsData = procesosResponse.data;
        const rawResults = Array.isArray(rawResultsData) ? rawResultsData : (rawResultsData ? [rawResultsData] : []);


        const planillas = rawResults.map(proceso => ({
            orden_proceso: proceso.planilla || null,
        }));

        const ordenesExistentes = await ProcesoHeader.findAll({
            where: { gen_orden_produccion: planillas.map(p => p.orden_proceso) },
            attributes: ['gen_orden_produccion'],
            raw: true
        });

        const ordenesYaSincronizadas = new Set(
            ordenesExistentes.map(o => o.gen_orden_produccion)
        );

        const planillasPendientes = rawResults.filter(
            proceso => !ordenesYaSincronizadas.has(proceso.planilla)
        );

        if (!planillasPendientes.length) {
            return res.status(200).json({
                message: 'No hay nuevas órdenes de proceso para sincronizar.'
            });
        }

        // --- CONSULTAR MB51 ---

        const MAX_CONCURRENT = 5;
        const movimientosPorPlanilla = new Map();

        for (let i = 0; i < planillasPendientes.length; i += MAX_CONCURRENT) {
            const lote = planillasPendientes.slice(i, i + MAX_CONCURRENT);

            await Promise.all(
                lote.map(async (proceso) => {
                    const planilla = proceso.planilla;
                    try {
                        // console.log(`solicitando MB51 para ${planilla}`);
                        const lista = await consultarMB51(planilla, tokenSAP); // ahora es array

                        // DEBUG fuerte para entender por qué quedó vacío
                        console.log(`RESPUESTA PARA ${planilla}: tipo=${Array.isArray(lista) ? 'array' : typeof lista}, length=${lista?.length}`);
                        if (lista && lista.length > 0) {
                            movimientosPorPlanilla.set(planilla, { proceso, movimientos: lista });
                            console.log(`Guardado ${planilla} -> ${lista.length} movimientos`);
                        } else {
                            console.log(`(vacío) No hay movimientos para ${planilla}`);
                        }
                    } catch (err) {
                        console.error(`Error MB51 ${planilla}:`, err?.message || err);
                        movimientosPorPlanilla.set(planilla, { proceso, movimientos: null, error: err?.message });
                    }
                })
            );
        }

        // --- ENRIQUECER: obtener detalles CHARG y combinar (helper) ---
        // Construir estructura esperada por enriquecerDatosConDetallesLote:
        // movimientosPorPlanilla => Map(planilla -> { proceso, movimientos })
        const datosArrayParaEnriquecer = Array.from(movimientosPorPlanilla.entries())
            .map(([planilla, data]) => ({ planilla, proceso: data.proceso, movimientos: data.movimientos }));

        // Llamada al helper: devuelve array [{ planilla, proceso, movimientos: [..con detalle..] }]
        const datosEnriquecidos = await enriquecerDatosConDetallesLote(datosArrayParaEnriquecer, tokenSAP, RFC_BASE);

        if (!datosEnriquecidos || !datosEnriquecidos.length) {
            return res.status(200).json({ message: 'No se generaron datos enriquecidos para insertar.' });
        }

        // --- PREPARAR E INSERTAR EN DB EN UNA TRANSACCIÓN ---
        if (!sequelize) {
            console.warn('Sequelize no encontrado en ProcesoHeader.sequelize — insertaremos sin transacción.');
        }

        const doTransaction = async (t) => {
            // prepararDatosParaInsercion -> { headers, entradas, salidas }
            const { headers, entradas, salidas } = await prepararDatosParaInsercion(datosEnriquecidos, t);

            // Si no hay headers válidos
            if (!headers.length) {
                return { headersInsertados: 0, entradasInsertadas: 0, salidasInsertadas: 0 };
            }

            // Insertar headers (bulkCreate)
            const headersInsertados = await ProcesoHeader.bulkCreate(headers, { transaction: t, returning: true });

            // Map planilla -> id header insertado
            const headerIdMap = headersInsertados.reduce((map, h) => {
                map[h.gen_orden_produccion] = h.gen_idtbl_header_op;
                return map;
            }, {});

            // Asignar FK y eliminar 'planilla' auxiliar
            entradas.forEach(e => {
                e.gen_fk_tbl_header_op_idtbl_header_op = headerIdMap[e.planilla];
                delete e.planilla;
            });

            salidas.forEach(s => {
                s.gen_fk_tbl_header_op_idtbl_header_op = headerIdMap[s.planilla];
                delete s.planilla;
            });

            // Bulk insert entradas y salidas
            if (entradas.length) await ProcesoIn.bulkCreate(entradas, { transaction: t });
            if (salidas.length) await ProcesoOut.bulkCreate(salidas, { transaction: t });

            // Logs
            await Logs.create({
                gen_fk_tbl_usuario_gen_id_usuario: null,
                gen_accion: 'SINCRONIZACIÓN CEREZA',
                gen_descripcion: `Sincronizados ${headersInsertados.length} procesos de cereza desde API externa`,
                gen_fecha: new Date()
            }, { transaction: t });

            return {
                headersInsertados: headersInsertados.length,
                entradasInsertadas: entradas.length,
                salidasInsertadas: salidas.length
            };
        };

        let resultadoInsercion;
        if (sequelize) {
            // Con transacción
            resultadoInsercion = await sequelize.transaction(async (t) => {
                return await doTransaction(t);
            });
        } else {
            // Sin transacción (no recomendado en prod)
            resultadoInsercion = await doTransaction(null);
        }

        return res.json({
            success: true,
            message: "Sincronización completada",
            totalPlanillasProcesadas: datosEnriquecidos.length,
            resumen: {
                headersInsertados: resultadoInsercion.headersInsertados,
                entradasInsertadas: resultadoInsercion.entradasInsertadas,
                salidasInsertadas: resultadoInsercion.salidasInsertadas
            },
            omitidos: resultadoInsercion.omitidos // Make sure doTransaction returns omitidos
        });

    } catch (error) {
        console.error('sincronizar_procesos_etl_cerezas error:', error);
        return res.status(500).json({
            error: error?.message || error
        });
    }
};

module.exports = { sincronizar_rs_reina_sur, sincronizar_procesos_etl_cerezas };